﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Data;
using System.Data.SqlClient;

namespace CDNModule
{
    /// <summary>
    /// Класс хронящий результат запроса
    /// </summary>
    public class TODBC_Array
    {
        /// <summary>
        /// двумерный массив полученной таблицы при загрузке DBF-файла или выполнения SQL-запроса
        /// </summary>
        private Dictionary<int, object[]> queryResultTable;
        /// <summary>
        /// массив имен полей полученных при загрузке DBF-файла или  выполнения SQL-запроса
        /// </summary>
        private string[] fieldsArray;
        /// <summary>
        /// хранит позицию читаемой строки
        /// </summary>
        private int currentRowPosition = 0;
        /// <summary>
        /// хранит количество строк в массиве
        /// </summary>
        private int totalRowCount = 0;
        /// <summary>
        /// Хронит состояние об "был ли получен результат со строками"
        /// </summary>
        private bool isSelected = false;
        /// <summary>
        /// хранит позицию последнего искомого поля
        /// </summary>
        private int lastSearchedFieldPosition = -1;
        /// <summary>
        /// хранит имя последнего искомого поля
        /// </summary>
        private string lastSearchedFieldName = "";

        /// <summary>
        /// Возвращает 'true', если были получены строки
        /// </summary>
        public bool IsSelRecord { get { return isSelected; } }
        /// <summary>
        /// Возвращает кол-во строк
        /// </summary>
        public int CountRecord { get { return totalRowCount; } }
        /// <summary>
        /// Возвращает кол-во полей
        /// </summary>
        public int CountField { get { return fieldsArray.Length; } }

        /// <summary>
        /// Возвращает имя поля или текст ошибки
        /// </summary>
        /// <param name="index">Номер поля</param>
        /// <returns>Возвращает имя поля или текст ошибки</returns>
        public string FieldName(int index)
        {
            if (isSelected)
            {
                if (fieldsArray.Length > index && index >= 0)
                { return fieldsArray[index]; }
                else return ("ERROR: Номера поля '" + index.ToString() + "' не существует");
            }
            else return "ERROR: Запрос не является 'Select'";
        }
        /// <summary>
        /// Возвращает индекс поля или -1 в случае,
        /// если поля с искомым именем небыло найдено
        /// </summary>
        /// <param name="Name">Имя поля</param>
        /// <returns>Возвращает индекс поля или -1 в случае ошибки</returns>
        public int FieldIndex(string Name)
        {
            IsField(Name);
            return lastSearchedFieldPosition;
        }
        /// <summary>
        /// Возвращает результат поиска поля.
        /// Если поля не существует, то вернет 'false'
        /// </summary>
        /// <param name="Name">Имя поля</param>
        /// <returns>Возвращает результат поиска поля.</returns>
        public bool IsField(string Name)
        {
            Name = Name.ToLower();
            if (isSelected)
            {
                if (lastSearchedFieldName != Name)
                {
                    lastSearchedFieldName = Name;
                    try
                    { lastSearchedFieldPosition = fieldsArray.ToList().IndexOf(Name); }
                    catch
                    { lastSearchedFieldPosition = -1; }
                }
                return (lastSearchedFieldPosition >= 0);
            }
            else return false;
        }

        /// <summary>
        /// возвращает значения указанной строки
        /// </summary>
        /// <param name="row">номер строки</param>
        public object[] RowToAr(int row)
        {
            if (isSelected && totalRowCount > row && row >= 0)
            { return queryResultTable[row]; }
            else return null;
        }
        /// <summary>
        /// Возвращиет значения текущей строки
        /// </summary>
        public object[] RowToAr()
        {
            if (isSelected && !EOF())
            { return queryResultTable[currentRowPosition]; }
            else return null;
        }

        /// <summary>
        /// возвращает текстовые значения указанной строки
        /// </summary>
        /// <param name="row">Номер строки</param>
        public string[] RowToAr_s(int row)
        {
            if (isSelected && totalRowCount > row && row >= 0)
            {
                string[] s = new string[fieldsArray.Length];
                object obj = null;

                for (var i = 0; i < fieldsArray.Length; i++)
                {
                    obj = queryResultTable[row][i];
                    s[i] = ((obj != null) ? obj.ToString() : "");
                }
                return s;
            }
            else return null;
        }
        /// <summary>
        /// возвращает текстовые значения текущей строки
        /// </summary>
        public string[] RowToAr_s()
        {
            if (isSelected && !EOF())
            {
                object obj = null;
                string[] s = new string[fieldsArray.Length];

                for (var i = 0; i < fieldsArray.Length; i++)
                {
                    obj = queryResultTable[currentRowPosition][i];
                    s[i] = ((obj != null) ? obj.ToString() : "");
                }
                return s;
            }
            else return null;
        }

        /// <summary>
        /// Возвращает хранящийся двумерный массив
        /// </summary>
        /// <value>Возвращает двумерный массив</value>
        public Dictionary<int, object[]> GetArray
        { get { return queryResultTable; } }
        /// <summary>
        /// Создает массив строк. Применяется при получении таблици из SQL-запроса
        /// </summary>
        /// <param name="read">результат при запросе</param>
        public TODBC_Array(SqlDataReader read)
        {
            if (read.HasRows)
            {
                isSelected = true;
                totalRowCount = read.FieldCount;//!!!!!!!!!!!! ХЗ зачем... Экономия, наверно...

                fieldsArray = new string[totalRowCount];
                for (int fieldIndex = 0; fieldIndex < totalRowCount; fieldIndex++)
                { 
                    fieldsArray[fieldIndex] = read.GetName(fieldIndex).ToLower(); 
                }

                queryResultTable = new Dictionary<int, object[]>();
                int rowCounter = 0;
                while (read.Read())
                {
                    queryResultTable[rowCounter] = new object[totalRowCount];
                    for (int fieldIndex = 0; fieldIndex < read.FieldCount; fieldIndex++)
                    { 
                        queryResultTable[rowCounter][fieldIndex] = read.GetValue(fieldIndex); 
                    }
                    rowCounter++;
                }
                totalRowCount = rowCounter;
            }
            read.Close();
            read.Dispose();
        }
        /// <summary>
        /// Хранит кол-во обработанных строк. Применяется во
        /// всех запросов, которые в качестве результатов возвращают не таблицу
        /// </summary>
        /// <param name="read">число обработанных строк</param>
        public TODBC_Array(int read) { totalRowCount = read; }

        /// <summary>
        /// Загружает DBF-файл в таблицу
        /// </summary>
        /// <param name="DBFFileName">Путь и имя DBF-файла</param>
        public TODBC_Array(string DBFFileName)
        {
            System.IO.FileStream FS = new System.IO.FileStream(DBFFileName, System.IO.FileMode.Open);
            byte[] buffer = new byte[4]; // Кол-во записей: 4 байтa, начиная с 5-го
            FS.Position = 4; FS.Read(buffer, 0, buffer.Length);
            //int RowsCount
            totalRowCount = buffer[0] + (buffer[1] * 0x100) + (buffer[2] * 0x10000) + (buffer[3] * 0x1000000);
            buffer = new byte[2]; // Кол-во полей: 2 байтa, начиная с 9-го
            FS.Position = 8; FS.Read(buffer, 0, buffer.Length);
            int offset = (buffer[0] + (buffer[1] * 0x100));
            int FieldCount = (((buffer[0] + (buffer[1] * 0x100)) - 1) / 32) - 1;

            fieldsArray = new string[FieldCount]; // Массив названий полей

            string[] FieldType = new string[FieldCount]; // Массив типов полей
            byte[] FieldSize = new byte[FieldCount]; // Массив размеров полей
            byte[] FieldDigs = new byte[FieldCount]; // Массив размеров дробной части
            buffer = new byte[32 * FieldCount]; // Описание полей: 32 байтa * кол-во, начиная с 33-го
            FS.Position = 32; FS.Read(buffer, 0, buffer.Length);
            int FieldsLength = 0;
            for (int i = 0; i < FieldCount; i++)
            {
                // Заголовки
                fieldsArray[i] = System.Text.Encoding.Default.GetString(buffer, i * 32, 10).TrimEnd(new char[] { (char)0x00 });
                fieldsArray[i] = fieldsArray[i].Split((char)0x00)[0].ToLower();
                //fl[i] = fl[i].Substring(0, fl[i].IndexOf((char)0x00)).ToLower();
                FieldType[i] = "" + (char)buffer[i * 32 + 11];
                FieldSize[i] = buffer[i * 32 + 16];
                FieldDigs[i] = buffer[i * 32 + 17];
                FieldsLength = FieldsLength + FieldSize[i];
            }

            FS.ReadByte(); // Пропускаю разделитель схемы и данных
            System.Globalization.DateTimeFormatInfo dfi = new System.Globalization.CultureInfo("en-US", false).DateTimeFormat;
            System.Globalization.NumberFormatInfo nfi = new System.Globalization.CultureInfo("en-US", false).NumberFormat;
            buffer = new byte[FieldsLength];
            isSelected = totalRowCount > 0;
            FS.Position = offset;
            if (isSelected)
            {
                queryResultTable = new Dictionary<int, object[]>();
                for (int j = 0; j < totalRowCount; j++)
                {
                    FS.ReadByte(); // Пропускаю стартовый байт элемента данных
                    FS.Read(buffer, 0, buffer.Length);
                    object[] R = new object[FieldCount];
                    int Index = 0;
                    for (int i = 0; i < FieldCount; i++)
                    {
                        string l = System.Text.Encoding.GetEncoding(866).GetString(buffer, Index, FieldSize[i]).TrimEnd(new char[] { (char)0x00 }).TrimEnd(new char[] { (char)0x20 });
                        Index = Index + FieldSize[i];
                        if (l != "")
                            switch (FieldType[i])
                            {
                                case "L": R[i] = l == "T" ? true : false; break;
                                case "D": R[i] = DateTime.ParseExact(l, "yyyyMMdd", dfi); break;
                                case "N":
                                    {
                                        if (FieldDigs[i] == 0)
                                            R[i] = int.Parse(l, nfi);
                                        else try
                                            { R[i] = decimal.Parse(l, nfi); }
                                            catch { throw new Exception("[" + i.ToString() + "]>{" + Index.ToString() + "} '" + l + "' = " + nfi.ToString()); }

                                        break;
                                    }
                                case "F": R[i] = double.Parse(l, nfi); break;
                                default: R[i] = l; break;
                            }
                        else
                            R[i] = "";
                    }
                    queryResultTable[j] = R;
                }
            }
            FS.Close();
        }

        /// <summary>
        /// Возвращает новый экземпляр TODBC_Array, в который помещает строки по указанному фильтру
        /// </summary>
        /// <param name="Filter">фильтр, где ключ=имени полю, а значение по ключю=искомое значение в этом поле</param>
        /// <param name="sel">способ отсеивания</param>
        /// <param name="low">если поставить в "True", то регистр букв при поиске будет игнорироваться</param>
        public TODBC_Array SelectArray(Dictionary<string, string> Filter, SelectAr sel, bool low)
        {
            if (Filter != null && isSelected)
            {
                TODBC_Array ar = new TODBC_Array(0);
                ar.queryResultTable = new Dictionary<int, object[]>();
                ar.fieldsArray = fieldsArray;
                ar.totalRowCount = 0;
                for (int i = 0; i < totalRowCount; i++)
                {
                    bool bl = true;
                    foreach (string k in Filter.Keys)
                    {
                        if (IsField(k))
                        {
                            string s1 = Cells(i, k).ToString();
                            string s2 = Filter[k];
                            if (low)
                            {
                                s1 = s1.ToLower();
                                s2 = s2.ToLower();
                            }
                            switch (sel)
                            {
                                case SelectAr.Equ:
                                    bl = bl && (s1 == s2);
                                    break;
                                case SelectAr.All:
                                    bl = bl && (s1.IndexOf(s2) >= 0);
                                    break;
                                case SelectAr.Left:
                                    bl = bl && (s1.IndexOf(s2) == 0);
                                    break;
                                case SelectAr.Right:
                                    bl = bl && s1.IndexOf(s2) <= s1.Length - s2.Length;
                                    break;
                                case SelectAr.Center:
                                    int ii = s1.IndexOf(s2);
                                    bl = bl && ii < s1.Length - s2.Length && ii > 0;
                                    break;
                            }

                        }
                        else { bl = false; }
                    }
                    if (bl)
                    {
                        ar.queryResultTable[ar.totalRowCount] = queryResultTable[i];
                        ar.totalRowCount++;
                    }
                }
                ar.isSelected = ar.totalRowCount > 0;
                return ar;
            }
            else { return null; }
        }

        /// <summary>
        /// Позволяет извлекать данные из полученной
        /// таблице
        /// </summary>
        /// <param name="row">Номер строки</param>
        /// <param name="col">Имя поля</param>
        /// <returns>Возвращает данные из таблици</returns>
        public object Cells(int row, string col)
        {
            if (isSelected)
            {
                if (totalRowCount > row)
                {
                    if (IsField(col))
                    { return queryResultTable[row][lastSearchedFieldPosition]; }
                    else return ("ERROR: Имя поля '" + col + "' не существует");
                }
                else return ("ERROR: Номер строки '" + row.ToString() + "' не существует");
            }
            else return "ERROR: Запрос не является 'Select'";
        }
        /// <summary>
        /// Позволяет извлекать данные из полученной
        /// таблице
        /// </summary>
        /// <param name="row">Номер строки</param>
        /// <param name="col">Номер поля</param>
        /// <returns>Возвращает данные из таблици</returns>
        public object Cells(int row, int col)
        {
            if (isSelected)
            {
                if (totalRowCount > row && row >= 0)
                {
                    if (fieldsArray.Length > col)
                    { return queryResultTable[row][col]; }
                    else return ("ERROR: Номера поля '" + col.ToString() + "' не существует");
                }
                else return ("ERROR: Номер строки '" + row.ToString() + "' не существует");
            }
            else return "ERROR: Запрос не является 'Select'";
        }

        /// <summary>
        /// Позволяет устанавливать и получать позицию курсора
        /// в текущей таблици.
        /// </summary>
        public int Position
        {
            get { return currentRowPosition; }
            set { currentRowPosition = value; }
        }
        /// <summary>
        /// Устанавливает курсор в начало таблици
        /// </summary>
        /// <returns></returns>
        public int First() { currentRowPosition = 0; return currentRowPosition; }
        /// <summary>
        /// Устанавливает курсор в конец таблици
        /// </summary>
        /// <returns></returns>
        public int Last() { currentRowPosition = totalRowCount; return currentRowPosition; }
        /// <summary>
        /// Перемещает курсор на 1 строку дальше
        /// </summary>
        /// <returns></returns>
        public int Next() { currentRowPosition++; return currentRowPosition; }
        /// <summary>
        /// Перемещает курсор на 'inc' строк дальше
        /// </summary>
        /// <param name="inc">шаг перемещения курсора</param>
        /// <returns></returns>
        public int Next(int inc) { currentRowPosition += inc; return currentRowPosition; }
        /// <summary>
        /// Перемещает курсор на 1 строку назад
        /// </summary>
        /// <returns></returns>
        public int Prior() { currentRowPosition--; return currentRowPosition; }
        /// <summary>
        /// Перемещает курсор на 'dec' строк назад
        /// </summary>
        /// <param name="dec">шаг перемещения курсора</param>
        /// <returns></returns>
        public int Prior(int dec) { currentRowPosition -= dec; return currentRowPosition; }
        /// <summary>
        /// Возвращает 'true' если курсор вышел за рамки массива
        /// </summary>
        /// <returns></returns>
        public bool EOF() { return ((currentRowPosition < 0) || (currentRowPosition >= totalRowCount)); }
        /// <summary>
        /// Возвращает значение поля в текущей строке 
        /// (указанной курсором)
        /// </summary>
        /// <param name="Index">Номер поля</param>
        /// <returns></returns>
        public object Field(int Index)
        {
            if (isSelected)
            {
                if (!EOF())
                {
                    if (fieldsArray.Length > Index)
                    { return queryResultTable[currentRowPosition][Index]; }
                    else return ("ERROR: Номера поля '" + Index.ToString() + "' не существует");
                }
                else return "ERROR: Указатель вышел за пределы массива";
            }
            else return "ERROR: Запрос не является 'Select'";
        }
        /// <summary>
        /// Возвращает значение поля в текущей строке 
        /// (указанной курсором)
        /// </summary>
        /// <param name="Name">Имя поля</param>
        /// <returns></returns>
        public object Field(string Name)
        {
            if (isSelected)
            {
                if (!EOF())
                {
                    if (IsField(Name))
                    {
                        return queryResultTable[currentRowPosition][lastSearchedFieldPosition];
                    }
                    else return ("ERROR: Имя поля '" + Name + "' не существует");
                }
                else return "ERROR: Указатель вышел за пределы массива";
            }
            else return "ERROR: Запрос не является 'Select'";
        }

        /// <summary>
        /// Возвращает значение поля, преобразуя его в тип string,в текущей строке 
        /// (указанной курсором)
        /// </summary>
        /// <param name="Name">Имя поля</param>
        /// <returns></returns>
        public string Field_s(string Name)
        {
            if (isSelected && !EOF())
            {
                object obj = Field(Name);
                return ((obj != null) ? obj.ToString() : "");
            }
            else { return ""; }
        }

        /// <summary>
        /// Итератор позволяющий пробежаться по всем строкам
        /// </summary>
        /// <param name="ODBCEach">Делегат итератора</param>
        public void EachRow(TODBCArrayEachRow ODBCEach)
        {
            if (ODBCEach != null && isSelected)
            {
                for (int i = 0; i < totalRowCount; i++)
                    ODBCEach(i, queryResultTable[i]);
            }
        }
        /// <summary>
        /// Итератор позволяющий пробежаться по всей строке,
        /// на которую указывает курсор
        /// </summary>
        /// <param name="ODBCEach">Делегат итератора</param>
        public void EachCol(TODBCArrayEachCol ODBCEach)
        {
            if (ODBCEach != null && isSelected && !EOF())
            {
                for (int i = 0; i < fieldsArray.Length; i++)
                    ODBCEach(fieldsArray[i], queryResultTable[currentRowPosition][i]);
            }
        }
        /// <summary>
        /// Итератор позволяющий пробежаться по всем 
        /// строкам и всем полям этих строк
        /// </summary>
        /// <param name="ODBCEach">Делегат итератора</param>
        public void EachRowCol(TODBCArrayEachRowCol ODBCEach)
        {
            if (ODBCEach != null && isSelected)
            {
                for (int i = 0; i < totalRowCount; i++)
                    for (int j = 0; j < fieldsArray.Length; j++)
                        ODBCEach(i, fieldsArray[j], queryResultTable[i][j]);
            }
        }

        /// <summary>
        /// временный класс для формирования листа
        /// </summary>
        public class TODBC_Array_Item
        {
            /// <summary>
            /// массив хранящий строку (имяполя/его значение)
            /// </summary>
            private Dictionary<string, string> t = new Dictionary<string, string>();
            /// <summary>
            /// Создает строку для листа из TODBC_Array
            /// </summary>
            /// <param name="row">номер строки</param>
            /// <param name="AR">TODBC_Array из которово будет браться строка</param>
            public TODBC_Array_Item(int row, TODBC_Array AR)
            {
                if (AR.isSelected && AR.fieldsArray.Length > 0 && AR.totalRowCount > row && row >= 0)
                    for (int i = 0; i < AR.fieldsArray.Length; i++)
                    {
                        object obj = AR.queryResultTable[row][i];
                        t[AR.fieldsArray[i]] = ((obj == null) ? "" : obj.ToString());
                    }
            }
            /// <summary>
            /// Возвращает массив строки
            /// </summary>
            public Dictionary<string, string> Rows { get { return t; } }
        }

        /// <summary>
        /// Возвращает лист асоциативного массива преобразованного из текущей таблицы
        /// </summary>
        public List<TODBC_Array_Item> Items()
        {
            List<TODBC_Array_Item> l = new List<TODBC_Array_Item>();
            if (isSelected)
                for (int i = 0; i < totalRowCount; i++)
                { l.Add(new TODBC_Array_Item(i, this)); }
            return l;
        }
    }

}
