<?php

namespace App\Classes\Xml;

use ArrayAccess;
use Illuminate\Support\Arr;
use Illuminate\Support\Str;

/*
# *.root
#			:> User::UniversalXML::Node or nil
# *.to_s
#			:> String
# *.last
#			:> User::UniversalXML::Node
# *.first
#			:> User::UniversalXML::Node
#			:> Array of User::UniversalXML::Node
# *.odd nodes = nil
#			<: nodes ~ nil or Array of User::UniversalXML::Node
#			:> Array of User::UniversalXML::Node
# *.even nodes = nil
#			<: nodes ~ nil or Array of User::UniversalXML::Node
#			:> Array of User::UniversalXML::Node
# *.node? tagname
#			<: tagname ~ Symbol or String
#			:> Bool
#
# *.get_tags_name
#			:> Array of String
# *.get_nodes_by_tag _tag, children = true
#			<: _tag ~ String or Symbol
#			<: children ~ Bool
#			:> Array of User::UniversalXML::Node
# *.get_nodes_by_id _id, children = true
#			<: _id ~ String
#			<: children ~ Bool
#			:> Array of User::UniversalXML::Node
# *.get_nodes_is_attr param, children = true
#			<: param ~ Any
#			<: children ~ Bool
#			:> Array of User::UniversalXML::Node
# *.get_nodes_is_value value, param = nil, children = true
#			<: value ~ Any
#			<: param ~ nil or Any
#			<: children ~ Bool
#			:> Array of User::UniversalXML::Node
#
# *.xmlPath path, oldresult = nil, &block
#			<: path ~ StringPath
#			<: oldresult ~ nil or Array of User::UniversalXML::Node
# *.match_find nodes, elem1, op = nil, elem2 = nil
#			<: nodes ~ nil or Array of User::UniversalXML::Node
#			<: elem1 ~ String
#			<: op ~ nil or String of '~~', '~', '!', '<=', '>=', '>', '<', '='
#			<: elem2 ~ nil or String
#			:> Bool
#
# ================================
# =          StringPath          =
# ================================
#
# /       - разделитель уровней
# &&      - разделитель поиска на одном уровне AND
# ||      - разделитель поиска на одном уровне OR
# .       - самый верхний узел
# ..      - верхний узел
# *       - обход всех подузлов или все узлы, если последний элемент
# **      - рекурсивный обход всех подэлементов
# *.      - все конечные элементы
# **.     - все конечные элементы, рекурсивный обход
# name    - узлы по тегу
# #       - отсутствие id и name
# #name   - узлы по id или name
# #:name  - регистронезависимые узлы по id или name
# @       - отсутствие атрибутов
# @name   - аттрибут тега
# @:name  - регистронезависимый аттрибут тега
# :last   - последний элемент в результате
# :first  - первый элемент в результате
# :number - элемент результата
# :odd    - нечетные элементы результата
# :even   - четные элементы результата
# :text   - текст хранящийся в теге
# 'text'  - строка
# :'text' - строка
# 123     - число
# true
# false
#
# вычисления (где $ означает либо аттрибут либо спецсимвол/значение):
#
# $a = $b   - если $a равен $b
# $a ~ $b   - если в $a присутствует $b
# $a ~~ $b  - если в $a присутствует $b (с игнорированием регистров)
# $a ! $b   - если $a не равен $b
# $a < $b   - если $a меньше $b
# $a > $b   - если $a больше $b
# $a <= $b  - если $a меньше или равен $b
# $a >= $b  - если $a больше или равен $b


# =================================
# =        Node Interface:        =
# =================================
#
# ---------------------------------
# -        Constructors           -
# ---------------------------------
#
# ...= User::UniversalXML::Node.new _tag, up = nil, _id = nil
#			<: _tag ~ String or Symbol
#			<: up ~ nil or User::UniversalXML::Node
#			<: _id ~ nil or String
# ...= User::UniversalXML::Node.load file_name
#			<: file_name ~ String
#
# ---------------------------------
# -          Data R/W             -
# ---------------------------------
#
# *.tag
#			:> String
# *.prefix new_prefix = false
#			<: new_prefix ~ false or String or Symbol or nil
#			:> String
# *.full_tag
#			:> String
# *.id
#			:> String
# *.obj_id
#			:> String
# *.is_attr? name, islow = false
#			<: name ~ Any
#			<: islow ~ Bool
#			:> Bool
# *.attrs keys = true
#			<: keys ~ Bool
#			:> Hash or Array of keys
# *.attr key, value = nil
#			<: key ~ Any
#			<: value ~ Any
#			:> self
# *.attrs_count
#			:> Numeric
# *.is_attrs?
#			:> Bool
# *.[] key
#			<: key ~ Any
#			:> String
# *.[]= key, value
#			<: key ~ Any
#			<: value ~ Any
# *.text value = nil
#			<: value ~ String
#			:> String
# *.text= value
#			<: value ~ String
# *.is_text?
#			:> Bool
# *.tag= newTag
#			<: newTag ~ String or Symbol
# *.attrs_parse children = true
#			<: children ~ Bool
#			:> self
# *.attrs_to_s children = true
#			<: children ~ Bool
#			:> self
# *.is_attr_parse?
#			:> Bool
#
# ---------------------------------
# -          Object Node          -
# ---------------------------------
#
# *.clone children = true
#			<: children ~ Bool
#			:> User::UniversalXML::Node
# *.move_to node
#			<: node ~ User::UniversalXML::Node
#			:> self
# *.move_me_down step = 1
#			<: step ~ :last or Number
#			:> self
# *.move_me_up step = 1
#			<: step ~ :first or Number
#			:> self
# *.move_me step
#			<: step ~ :first or :up or :down or :last or Number
#			:> self
# *.delete_me
#			:> self
# *.edit &block
#			:> self
# *.save file_name
#			<: file_name ~ String
#			:> self
# *.to_xml encoding='UTF-8', version='1.0'
#			<: encoding ~ nil or String
#			<: version ~ String
#			:> String
# *.to_h
#			:> Hash
# *.tags_to_h
#			:> Hash
# *.to_text at_this = false, opts = nil, lvl = ''
#			<: at_this ~ Bool
#			<: opts ~ nil or Array of [:cdata, :esc, :nopad, :small_end]  # default = [:cdata, :esc]
#			<: lvl ~ String
#			:> String
#
# ---------------------------------
# -          Tree Node            -
# ---------------------------------
#
# *.parent
#			:> User::UniversalXML::Node or nil
# *.nodes
#			:> Array of User::UniversalXML::Node
# *.is_nodes?
#			:> Bool
# *.length
#			:> Numeric
# *.clear
#			:> self
# *.delete node
#			<: node ~ User::UniversalXML::Node
#			:> self
# *.move_down node, step = 1
#			<: node ~ Number or User::UniversalXML::Node
#			<: step ~ :last or Number
#			:> self
# *.move_up node, step = 1
#			<: node ~ Number or User::UniversalXML::Node
#			<: step ~ :first or Number
#			:> self
# *.move node, step
#			<: node ~ Number or User::UniversalXML::Node
#			<: step ~ :first or :up or :down or :last or Number
#			:> self
# *.create_node name, _id = nil, &block
#			<: name ~ String or Symbol
#			<: _id ~ nil or String
#			:> User::UniversalXML::Node
# *.each &block
#			:> self
# *.size
#			:> Integer
# *.last_node path
#			<: path ~ String
#			:> self or User::UniversalXML::Node
#
# ---------------------------------
# -     Find or Filter Nodes      -
# ---------------------------------
#
# *.filter path = nil, &block
#			<: path ~ nil or StringPath
#			:> User::UniversalXML::Node
# *.find_id _id
#			<: _id ~ String
#			:> nil or self or User::UniversalXML::Node
#
# ---------------------------------
# -            Other              -
# ---------------------------------
#
# *.index node
#			<: node ~ User::UniversalXML::Node
#			:> Integer
# *.method_missing method, *args
#			:> User::UniversalXML::Node or String or self or nil
#
*/
class XmlNode implements ArrayAccess {
    /*
    # is_nodes?
    # parent
    # nodes
    # is_text?
    # is_attrs?
    # is_attr?
    # id
    # tag
    # full_tag
    # attrs
    # length
    # each
    # text
    */

    const PARAM_QUOTES = ['"',"'",'`'];

    static function parse_value($str){
        if($str == '')
            return null;
        elseif($str == 'empty')
            return '';
        elseif(preg_match("/^\d+\.\d+$/", $str))
            return (float) $str;
        elseif(preg_match("/^\d+$/", $str))
            return (int) $str;
        elseif(Str::lower($str) == 'true')
            return true;
        elseif(Str::lower($str) == 'false')
            return false;
        else
            return $str;
    }

    static function value_to_s($value){
        if(is_null($value))
            return '';
        elseif($value === '')
            return 'empty';
        else
            return ''.$value;
    }

    static function load($file_name){
        return static::parse_xml(file_get_contents($file_name));
    }

    /** @return XmlNode */
    static function parse_xml($text){
        $tags = explode('<', $text);
        array_shift($tags);
        /** @var XmlNode? $x */
        $x = null;
        $close_node = true;
        try{
        while(!empty($tags)){
            $tag = array_shift($tags);
            $sym = $tag[0] ?? '';
            switch($sym){
                #<?...
                case '?': break;

                #<!...
                #<![CDATA[...
                #<!--...
                case '!':
                    $tag = trim($tag);
                    if(substr($tag,0,7) == '![CDATA['){
                        $indx = -1;
                        foreach($tags as $i => $a)
                            if(strrpos(']]>',$a) !== false){
                                $indx = $i;
                                break;
                            }
                        $tmp = array_slice($tags, 0, $indx + 1);
                        $tag = '<' + implode('<',$tmp);
                        array_splice($tags, 0, $indx);
                        $len = strlen($tag);
                        if(substr($tag, $len-3) == ']]>')
                            $tag = substr($tag, 0, $len -3);
                        if($x)
                            $x->text($tag);
                        else
                            return $tag;
                    } elseif(substr($tag, 0, 3) == '!--'){
                        $tag = '';
                        while(!empty($tags) && substr($tag,strlen($tag)-3) != '-->')
                            $tag = array_shift($tags);
                    }
                    break;

                #</...
                case '/':
                    if(is_null($x->parent())) return $x;
                    $tag = trim($tag);
                    $tag = substr($tag, 1, strlen($tag)-2);
                    while($x->parent() && $x->full_tag() != $tag){
                        $x = $x->parent();
                    }

                    if(is_null($x->parent()))
                        return $x;
                    $x = $x->parent();
                    break;

                default: if(empty(trim($tag))){
                    if(!is_null($x)) $x->text($tag);
                } else {
                    $close = strrpos($tag, '>');
                    if($close === false){
                        $indx = -1;
                        foreach($tags as $i => $a)
                            if(strrpos($a, '>') !== false){
                                $indx = $i;
                                break;
                            }
                        $tag = $tag.'<'.implode('<', array_splice($tags, 0, $indx + 1));
                        array_splice($tags, 0, $indx);
                    }
                    $tag = trim($tag);
                    $close = strrpos($tag, '>');
                    if($close){
                        $text = substr($tag, $close+1);
                        if($tag[$close-1] == '/'){
                            $close -= 1;
                            $close_node = true;
                        }else
                            $close_node = false;
                        $tag = trim(substr($tag, 0, $close));
                    }else
                        $text = '';

                    if(empty($tag)){
                        if(!is_null($x)) $x->text($text);
                    } else {
                        $params = explode(' ',$tag);
                        $tagname = array_shift($params);
                        if(!empty($tagname)){
                            $x = is_null($x) ? new XmlNode($tagname) : $x->create_node($tagname);
                            $x->text($text);

                            while(!empty($params)){
                                $param = array_shift($params);
                                $e = strpos($param, '=');
                                if($e){
                                    list($key, $value) = explode('=', $param);
                                    if(in_array($value[0],self::PARAM_QUOTES) && $value[0] != $value[strlen($value)-1]){
                                        $sym = $value[0];
                                        while(in_array($value[strlen($value)-1], self::PARAM_QUOTES) && $value[strlen($value)-1] == $sym)
                                            $value .= ' ' + array_shift($params);
                                    }
                                    if($value[0] == $value[strlen($value)-1] && in_array($value[0], self::PARAM_QUOTES))
                                        $value = substr($value, 1, strlen($value)-2);
                                    $x[$key] = strtr($value, ['&quot;'=>'"','&apos;'=>"'", '&lt;'=>'<', '&gt;'=>'>', '&amp;'=>'&']);
                                } else
                                    $x[$param] = $param;
                            }
                        }
                        if($close_node) $x = $x->parent();
                    }
                }
            }
        }}catch(\Exception $e){
            var_dump("XmlNode - ошибка при парсинге xml - \n".$e->getMessage()."\nNoParse:\n".implode('<',$tags));
        }
        return $x;
    }

    private
        $_parse_value = false,
        $_nodes = [],
        $_parent = null,
        $_tagName, $_prefix, $_params, $_text;

    /**
     * @param string $tag
     * @param XmlNode|null $up
     * @param string|null $id
     * @param $func => function(XmlNode $node)
     * @return XmlNode
     */
    function __construct($tag, $up = null, $id = null, $func = null){
        $this->_parse_value = false;
        $tag = explode(':',$tag);
        if(count($tag) > 1){
            $this->_prefix = $tag[0];
            $this->_tagName = $tag[1];
        } else {
            $this->_prefix = null;
            $this->_tagName = $tag[0];
        }
        $this->_params = is_null($id) ? [] : ['id'=>$id];
        $this->_nodes = [];
        $this->_parent = $up;
        $this->_text = '';
        if(!is_null($func))
            $func($this);
    }

    function clone($children = true){
        $res = new XmlNode($this->_tagName);

    }

    function tag(){ return $this->_tagName; }
    function prefix($newPrefix = false){
        if($newPrefix !== false) $this->_prefix = $newPrefix;
        return $this->_prefix;
    }
    function full_tag(){
        if($this->_prefix)
            return $this->_prefix.':'.$this->_tagName;
        return $this->_tagName;
    }
    function setTag($newTag){
        if($newTag){
            $newTag = explode(':', $newTag);
            if(count($newTag) > 1){
                $this->_prefix = $newTag[0];
                $this->_tagName = $newTag[1];
            } else {
                $this->_prefix = null;
                $this->_tagName = $newTag[0];
            }
        }
    }

    function id(){
        if(isset($this->_params['id']))
            return $this->_params['id'];
        elseif(isset($this->_params['name']))
            return $this->_params['name'];
        return null;
    }
    function name(){ return $this->id(); }

    function obj_id(){
        if($this->_parent){
            $path = $this->_parent->obj_id();
            $result = ''.$this->_parent->index($this);
        } else {
            $path = '';
            $result = '';
        }
        return (empty($path) ? '' : $path.'/').$this->_tagName.(empty($result) ? '' : '#'.$result);
    }

    function has_attr($attrName, $isLow = false){
        if($isLow)
            return in_array(Str::lower($attrName), array_map(fn($str)=>Str::lower($str),array_keys($this->_params)));
        else
            return isset($this->_params[$attrName]);
    }

    function attrs($keys = true){
        return $keys ? array_keys($this->_params) : $this->_params;
    }

    function attrs_count(){return count($this->_params); }
    function is_attrs(){return !empty($this->_params); }
    function is_nodes(){return !empty($this->_nodes); }
    function is_text(){ return !empty($this->_text); }
    function is_attr_parse(){ return $this->_parse_value; }
    function nodes(){ return $this->_nodes; }
    function parent(){ return $this->_parent; }
    function length(){ return count($this->_nodes); }
    function index($node){ return array_search($node, $this->_nodes); }
    function find_id($id){
        if(isset($this->_params['id']) && $this->_params['id'] == $id)
            return $this;
        if($this->_nodes)
            foreach($this->_nodes as $node)
                if($node['id'] == $id)
                    return $node;
        return null;
    }

    // implementation interface ArrayAccess // использование [] у экземпляра класса
    function offsetExists($offset):bool{ return isset($this->_params);}
    function offsetGet($offset){ return $this->_params[$offset] ?? null; } // xml[key]
    function offsetSet($offset, $value): void { // xml[key] = value
        if(!$this->_parse_value) $value = ''.$value;
        if(is_null($offset))
            $this->_params[] = $value;
        else
            $this->_params[$offset] = $value;
    }
    function offsetUnset($offset): void{ unset($this->_params[$offset]); }
    //end implementation interface ArrayAccess

    function attrs_parse($children = true){
        $this->_parse_value = true;
        foreach($this->_params as $key => &$value)
            $value = static::parse_value($value);
        if($children && !empty($this->_nodes))
            foreach($this->_nodes as $node)
                $node->attrs_parse(true);
        return $this;
    }

    function attrs_to_s($children = true){
        $this->_parse_value = true;
        foreach($this->_params as $key => &$value)
            $value = static::value_to_s($value);
        if($children && !empty($this->_nodes))
            foreach($this->_nodes as $node)
                $node->attrs_to_s(true);
        return $this;
    }

    function move(XmlNode $node, $step){
        if(is_numeric($node)){
            $indx = $node < 0 ? count($this->_nodes)+$node : $node;
            $node = $this->_nodes[$indx] ?? null;
        } else $indx = array_search($node, $this->_nodes);
        unset($this->_nodes[$indx]);
        if($step == 'first')
            $indx = 0;
        elseif($step == 'last')
            $indx = count($this->_nodes) - 1;
        elseif($step == 'up')
            $indx = $indx > 0 ? $indx - 1 : 0;
        elseif($step == 'down')
            $indx = $indx < count($this->_nodes) ? $indx + 1 : count($this->_nodes);
        elseif($step < 0)
            $indx = $step >= $indx ? $indx + $step : 0;
        elseif($step > 0)
            $indx = $indx + $step < count($this->_nodes) ? $indx + $step : count($this->_nodes) - 1;
        array_splice($this->_nodes, $indx, 0, $node);
        return $this;
    }

    function move_up(XmlNode $node, $step = 1){ return $this->move($node, is_numeric($step) ? -$step : $step); }
    function move_down(XmlNode $node, $step = 1){ return $this->move($node, $step); }
    function move_me_up($step = 1){
        if(!is_null($this->_parent)) $this->_parent->move_up($this, $step);
        return $this;
    }
    function move_me_down($step = 1){
        if(!is_null($this->_parent)) $this->_parent->move_down($this, $step);
        return $this;
    }
    function move_me($step){
        if(!is_null($this->_parent)) $this->_parent->move($this, $step);
        return $this;
    }
    function move_to(XmlNode $node){
        if(!is_null($this->_parent)) $this->_parent->delete($this);
        $this->_parent = $node;
        $node->_nodes[] = $this;
        return $this;
    }

    function clear(){
        $this->_nodes = [];
        return $this;
    }

    /**
     * @param XmlNode|Integer $node
     */
    function delete($node){
        $delIndx = is_numeric($node) ? $node : array_search($node, $this->_nodes);
        unset($this->_nodes[$delIndx]);
        $this->_nodes = array_values($this->_nodes);
        return $this;
    }
    function delete_me(){
        if(!is_null($this->_parent)) $this->_parent->delete($this);
        return $this;
    }

    function is_attr($key){
        return isset($this->_params[$key]);
    }

    function attr($key, $value = null){
        $this->_params[$key] = $this->_parse_value ? $value : ''.$value;
        return $this;
    }

    function text($value = null){
        if(!is_null($value)){
            $this->_text = trim(''.$value);
            return $this;
        }
        if(empty($this->_nodes))
            return $this->_text;
        return $this->to_text();
    }
    function setText($value){
        $this->_text = trim(''.$value);
        return $this;
    }

    function each($func){
        foreach($this->_nodes as $node)
            $func($node);
    }

    function create_node($name, $id = null, $func = null){
        $node = new XmlNode($name, $this, $id);
        $this->_nodes[] = $node;
        if(!is_null($func)) $func($node);
        return $node;
    }

    function text_node($name, $text){
        $node = new XmlNode($name, $this);
        $node->text($text);
        $this->_nodes[] = $node;
        return $node;
    }


    function __toString() {
        return 'XmlNode(#'.spl_object_id($this).')';
    }

    function to_s(){
        $attrs = [];
        if($this->_params)
            foreach($this->_params as $key=>$val)
                $attrs[] = $key.'="'.$val.'"';
        return 'Node(#'.spl_object_id($this).'):<'.$this->full_tag().'['.implode(',', $attrs).'] > ...'.count($this->_nodes).' elements';
    }



    // корневой узел (самый верхний узел XML-документа)
    function root(){
        $r = $this->_parent ?? $this;
        while( !is_null($r->_parent) )
            $r = $r->parent;
        return $r;
    }

    // возвращает все вложенные узлы, лежащие по нечетным индексам (1,3,5,7,...)
    function odd($nodes = null){
        if(is_null($nodes))
            $nodes = $this->_nodes;
        $result = [];
        foreach($nodes as $i => $node)
            if($i & 1 == 1)
                $result[] = $node;
        return $result;
    }

    // возвращает все вложенные узлы, лежащие по четным индексам (0,2,4,6,...)
    function even($nodes = null){
        if(is_null($nodes))
            $nodes = $this->_nodes;
        $result = [];
        foreach($nodes as $i => $node)
            if($i & 1 == 0)
                $result[] = $node;
        return $result;
    }

    // возвращает последний вложенный узел
    function last(){
        $ct = count($this->_nodes);
        if($ct == 0) return null;
        return $this->_nodes[$ct-1];
    }

    // возвращает первый вложенный узел
    function first(){
        if(empty($this->_nodes)) return null;
        return $this->_nodes[0];
    }

    // возвращает true, если вложенный узел с указанный именем тега существует, и false в противном случае
    function isNode($tagname){
        foreach($this->_nodes as $node)
            if($node->full_tag() == $tagname || $node->tag() == $tagname)
                return true;
        return false;
    }

    // медот пробегает по всем вложенным узлам, и возвращает все конечные теги в виде хеша их имен и значений
    function tags_to_h(){
        $res = [];
        foreach($this->_nodes as $node)
            if(!$node->is_nodes())
                $res[$node->tag()] = $node->text();
        return $res;
    }

    // возвращает все имена вложенных узлов/тегов
    function get_tags_name(){
        $result = [];
        foreach($this->_nodes as $node)
            if(!in_array($node->tag(),$result))
                $result[] = $node->tag();
        return $result;
    }

    function __clone(){
        if(!empty($this->_nodes))
            foreach($this->_nodes as &$node)
                $node = clone $node;
    }

    function filter($path = null, $block = null){
        $filterNodes = is_null($path) ? $this->_nodes : $this->xmlPath($path);
        $result = new XmlNode('FILTER');
        foreach($filterNodes as $node)
            /** @var XmlNode $node */
            if(!$block || $block($node))
                (clone $node)->move_to($result);
        return $result;
    }

    # возвращает все указанныые вложенные узлы, где
    #   _tag     - наименование тега, которые необходимо вернуть
    #   children - если true, то будет запущен рекурсивный поиск по всем вложенным узлам
    function get_nodes_by_tag($tag, $children = true){
        $result = array_filter($this->_nodes, fn($node)=>$node->tag() == $tag);
        if($children)
            foreach($this->_nodes as $node)
                $result = array_merge($result, $node->get_nodes_by_tag($tag));
        return $result;
    }

    # возвращает все указанныые вложенные узлы, где
    #   _id      - чему должен быть равен атрибут "id" или "name" у тега, которые необходимо вернуть
    #   children - если true, то будет запущен рекурсивный поиск по всем вложенным узлам
    function get_nodes_by_id($id, $children = true){
        $result = array_filter($this->_nodes, fn($node)=>$node['id'] == $id);
        if($children)
            foreach($this->_nodes as $node)
                $result = array_merge($result, $node->get_nodes_by_id($id));
        return $result;
    }

    # возвращает все указанныые вложенные узлы, где
    #   param    - наименование атрибута, который должен существовать у тега, которые необходимо вернуть
    #   children - если true, то будет запущен рекурсивный поиск по всем вложенным узлам
    function get_nodes_is_attr($param, $children = true){
        $result = array_filter($this->_nodes, fn($node)=>$node->is_attr($param));
        if($children)
            foreach($this->_nodes as $node)
                $result = array_merge($result, $node->get_nodes_is_attr($param));
        return $result;
    }

    # возвращает все указанныые вложенные узлы, где
    #   value    - значение которому должен быть равен атрибут тега, которые необходимо вернуть
    #   param    - наименование атрибута, который нужно проверить у тега, которые необходимо вернуть,
    #              если указать как nil, то будет проверяться все атрибуты тега, на соответствие значению
    #   children - если true, то будет запущен рекурсивный поиск по всем вложенным узлам
    function get_nodes_is_value($value, $param = null, $children = true){
        if($param)
            $result = array_filter($this->_nodes, fn($node)=>$node->is_attr($param) && $node[$param] == ($this->_parse_value ?  $value : ''.$value));
        else
            $result = array_filter($this->_nodes, function($node)use($param, $value){
                /** @var XmlNode $node */
                $attrs = $node->attrs(false);
                $res = false;
                foreach($attrs as $attr)
                    $res = $res || ($this->_parse_value ? $attr == $value : ''.$attr == ''.$value);
                return $res;
            });
        if($children)
            foreach($this->_nodes as $node)
                $result = array_merge($result, $node->get_nodes_is_value($value, $param));
        return $result;
    }

    function __get($name) {
        $res = $this->last_node([$name]);
        return $res !== $this ? $res : null;
    }

    function last_node($path){
        if(is_string($path))
            $path = explode('/', $path);
        $name = array_shift($path);

        if($name == '@'){
            $name = array_shift($path);
            $prefix = '';
            if(strpos(':',$name) !== false)
                list($prefix, $name) = explode(':',$name);
            if($name == $this->_tagName && $prefix == $this->_prefix)
                $name = array_shift($path);
        }
        $prefix = '';
        if($name && strpos(':',$name))
            list($prefix, $name) = explode(':', $name);
        if(!$name) return $this;
        $result = null;
        foreach($this->_nodes as $node)
            if($node->tag() == $name ?? $node->prefix() == $prefix)
                $result = $node;
        if($result && count($path) > 0)
            $result = $result->last_node($path);
        return $result ?? $this;
    }

    function to_xml($version='1.0', $encoding='UTF-8'){
        $res = [];
        if(!is_null($version))
            $res[] = "<?xml version=\"$version\" encoding=\"$encoding\"?>\n";
        $res[] = $text = $this->to_text(true);
        return implode('',$res);
    }

    function method_missing(string $method, ...$args){
        if(empty($args)){
            #поиск по id
            foreach($this->_nodes as $node)
                if($node->id() == $method)
                    return $node;

            #поиск по имени тега
            foreach($this->_nodes as $node)
                if($node->tag() == $method)
                    return $node;

            #поиск параметра по имени
            if(isset($this->_param[$method]))
                return $this->_params[$method];

            #=== регистронезависимый поиск ===
            $methodL = Str::lower($method);

            foreach($this->_nodes as $node)
                if(Str::lower($node->id()??'') == $methodL)
                    return $node;

            #поиск по имени тега
            foreach($this->_nodes as $node)
                if(Str::lower($node->tag()) == $method)
                    return $node;

            throw new \Exception("Method, param, id and TAG '$method' not found");
        }elseif(is_array($args[0]) && count($args) == 1){
            foreach($this->_nodes as $node)
                if($node->tag() == $method){
                    $result = true;
                    $params = $node->attrs(false);
                    foreach($args[0] as $key => $val){
                        $result = $result && isset($params[$key]);
                        if($result){
                            if(preg_match($val, '') !== false)
                                $result = $result && $val->preg_match($val, $params[$key]);
                            else
                                $result = $result && $params[$key] == $val;
                        }
                    }
                    if($result) return $node;
                }

            throw new \Exception("Method and TAG '$method' is filter '".$args[0]."' not found");
        } elseif(count($args) == 1){
            if(strpos('=', $method) !== false)
                #поиск параметра по имени
                $this->_params[substr($method, 0, strlen($method)-1)] = $args[0];
            else{
                $this->params[$method] = $args[0];
                return $this;
            }
        } else
            throw new \Exception("Method '$method(".implode(',', $args).") not found");
    }

    function save($file_name){
        file_put_contents($file_name, $this->to_xml());
        return $this;
    }

    function to_h(){
        $res = clone $this->_params;
        $res['NODE_TEXT'] = $this->_text;
        return $res;
    }

    private function esc($text, $full = true){
        $text = $this->_parse_value ? static::value_to_s($text) : $text;
        $res = strtr($text, ['&'=>'&amp;', '<'=>'&lt;', '>'=>'&gt;']);
        return $full ? strtr($res, ['"'=>'&quot;', "'"=>'&apos;']) : $res;
    }

    function to_text($at_this = false, $opts = null, $lvl = ''){
        #opts:  ['cdata', 'esc', 'nopad', 'small_end']
        if(!is_array($opts)) $opts = ['cdata', 'esc'];
        $res = [];
        if($at_this){
            $res[] = "$lvl<".$this->full_tag();
            if(!empty($this->_params)){
                $res[] = ' ';
                foreach($this->_params as $key=>$value)
                    $res[] = ' '.$key.'="'.(in_array('esc',$opts) ? $this->esc($value) : $value).'"';
            }
        }
        if($at_this && empty($this->_nodes) && empty($this->_text) && in_array('small_end', $opts))
            $res[] = '/>';
        elseif(empty($this->_nodes) && empty($this->_text)){
            if($at_this)
                $res[] = '><'.$this->full_tag().'/>';
            else
                return '';
        }elseif(empty($this->_nodes)){
            if($at_this) $res[] = '>';
            if(preg_match("/[<>\"'&]/", $this->_text)){
                if(in_array('cdata', $opts))
                    $res[] = "<![CDATA[".$this->_text."]]>";
                else
                    $res[] = $this->esc($this->_text, false);
            } else
                $res[] = $this->_text;
            if($at_this) $res[] = "</".$this->full_tag().">";
        } else {
            $lvlnext = !$at_this && in_array('nopad', $opts) ? '' : $lvl . "\t";
            if($at_this) $res[] = ">\n";
            $res[] = implode("\n",array_map(fn($node)=>$node->to_text(true, $opts, $lvlnext), $this->_nodes));
            if($at_this) $res = "\n$lvl</".$this->full_tag().">";
        }
        return implode('', $res);
    }

    # возвращает все указанныые вложенные узлы, где
    #   path      - строка, которая является путем и описанием тегов, которые необходимо вернуть, подробную работу с ней смотрите в справочнике по модулю
    #   oldresult - массив узлов, из которых необходимо произвести дальнейшее отсеивание по указанной строке поиска
    #   &block    - если указан, то в него будет переданы поочередно все найденные узлы результата
    function xmlPath($path, $oldresult = null, $block = null){
        $tmp = $path;
        if(empty($this->_nodes) && substr($path,0,1) != '.')
            return [];
        if(is_string($path))
            $path = explode('/', $path);



        $start = true;
        $tempAND = explode('&&',array_shift($path));
        $resultAND = $oldresult;

        foreach($tempAND as $sAND){
            $tempOR = explode('||', $sAND);
            $resultOR = [];
            foreach($tempOR as $sOR){
                $sOR = trim($sOR);
                switch($sOR){
                    case '.':  $tempRes = [$this->root()]; break;
                    case '..': $tempRes = is_null($resultAND) ? [$this->_parent] : array_map(fn($node)=>$node->parent(), $resultAND); break;
                    case '*':  $tempRes = is_null($resultAND) ? $this->_nodes : Arr::flatten(array_map(fn($node)=>$node->nodes(), $resultAND)); break;
                    case '**':
                        if(is_null($resultAND)){
                            if($this->is_nodes())
                                $tempRes = array_unique(array_merge($this->_nodes, Arr::flatten(array_map(fn($node)=>$node->xmlPath(['**']), $this->_nodes))));
                            else
                                $tempRes = [];
                        } else
                            $tempRes = Arr::flatten(array_map(fn($node)=>$node->xmlPath(['**']), $resultAND));
                        break;
                    case '*.':
                        if(is_null($resultAND))
                            $tempRes = $this->is_nodes() ? array_filter($this->_nodes, fn($node)=>!$node->is_nodes()) : [];
                        else
                            $tempRes = Arr::flatten(array_map(fn($node)=>$node->xmlPath(['*.']), $resultAND));
                        break;
                    case '**.':
                        if(is_null($resultAND))
                            $tempRes = $this->is_nodes()
                                ? array_merge(array_filter($this->_nodes, fn($node)=>!$node->is_nodes()), Arr::flatten(array_map(fn($node)=>$node->xmlPath(['**.']), $this->_nodes)))
                                : [];
                        else
                            $tempRes = Arr::flatten(array_map(fn($node)=>$node->xmlPath(['**.']), $resultAND));
                        break;
                    case ':last':
                        $tempRes = array_filter([...($start
                                ? (is_null($resultAND) ? [$this->last()] : array_map(fn($node)=>$node->last(),$resultAND) )
                                : ($resultAND[count($resultAND) - 1] ?? null))
                            ], fn($node)=>!is_null($node));
                        break;
                    case ':first':
                        $tempRes = array_filter([...($start
                                ? (is_null($resultAND) ? [$this->first()] : array_map(fn($node)=>$node->first(),$resultAND) )
                                : ($resultAND[0] ?? null))
                            ], fn($node)=>!is_null($node));
                        break;
                    case ':odd': $tempRes = array_unique(Arr::flatten($this->odd($resultAND))); break;
                    case 'even': $tempRes = array_unique(Arr::flatten($this->even($resultAND))); break;
                    default:
                        $args = [$sOR];
                        foreach(['~~','~','!','<=','>=','>','<','='] as $op)
                            if(strpos($sOR, $op) !== false){
                                $tmp = explode($op, $sOR);
                                $args = [$tmp[0], $op, $tmp[1]];
                                break;
                            }
                        if($start && !is_null($resultAND))
                            $tempRes = Arr::flatten(array_map(fn($node)=>$node->match_find(null, ...$args), $resultAND));
                        else
                            $tempRes = Arr::flatten($this->match_find($resultAND, ...$args));
                        break;
                }
                $resultOR = array_merge($resultOR, $tempRes);
            }
            $start = false;
            $resultAND = $resultOR;
        }
        //dd(array_unique($resultAND, SORT_REGULAR));
        if(empty($resultAND)) return [];

        $result = empty($path) ? array_unique($resultAND) : array_unique(Arr::flatten($this->xmlPath($path, array_unique($resultAND))));
        if($block)
            foreach($result as $node)
                $block($node);
        return $result;
    }

    # вспомогательный метод, который работает внутри xmlPath, и занимается сравниванием значений, подробнее можно почитать в справочнике по модулю
    #   _nodes - массив узлов, которые следует отфильтровать, по указанным правилам
    #   elem1  - правое правило фильтрации
    #   op     - оператор сравнения
    #
    #   elem2  - левое правило фильтрации
    function match_find($nodes, $elem1, $op = null, $elem2 = null){
        if(is_null($nodes))
            $nodes = $this->_nodes;
        if(empty($nodes))
            return [];

        $arTmp = [];
        if($elem1) $arTmp[] = $elem1;
        if($elem2) $arTmp[] = $elem2;

        //if($elem1 == ':text') dd($nodes, $arTmp);

        $resultElement = [];
        foreach($nodes as $index => $node){
            /** @var XmlNode $node */
            # получение значений по указанным правилам


            $temp = array_map(function($elem)use(&$node, $op, $index){
                /** @var XmlNode $node */
                $el2 = substr($elem, 0, 2);
                $len = strlen($elem);
                //dd($node->tag(), $elem);
                if($elem == ':number')
                    return $index;
                elseif($elem == ':text')
                    return $op ? $node->text() : ($node->is_text() ? true : null);
                elseif($elem == 'true')
                    return true;
                elseif($elem == 'false')
                    return false;
                elseif($elem == '@')
                    return !$node->is_attrs();
                elseif($el2 == '@:'){
                    $key = substr($elem, 2);
                    return $op ? ($node->is_attr($key, true) ? $node[Str::lower($key)] : null) : $node->is_attr($key, true);
                } elseif($elem[0] == '@'){
                    $key = substr($elem, 1);
                    return $op ? ($node->is_attr($key) ? $node[$key] : null) : $node->is_attr($key);
                } elseif($elem == '#')
                    return $node->id() ? null : $node;
                elseif($el2 == '#:'){
                    $temp = ($elem[2] == $elem[$len-1] && $elem[2] == "'") ? substr($elem,3, $len-4) : substr($elem, 2);
                    return $node->id() && Str::lower($node->id()) == Str::lower($temp) ? $node : null;
                } elseif($elem[0] == '#'){
                    $temp = ($elem[1] == $elem[$len-1] && $elem[1] == "'") ? substr($elem,2, $len-3) : substr($elem, 1);
                    return $node->id() == $temp ? $node : null;
                } elseif(preg_match("/^\d+(\.\d+)?$/", $elem))
                    return (float)$elem;
                elseif($elem[0] == $elem[$len-1] && $elem[0] == "'")
                    return substr($elem, 1, $len - 2);
                elseif($elem[0] == ':' && $elem[1] == $elem[$len-1] && $elem[1] == "'")
                    return Str::lower(substr($elem, 2, $len-3));
                elseif($node->tag() == $elem || $node->full_tag() == $elem){
                    return $node;
                }else
                    return null;
            }, $arTmp);

            # предварительное преобрасование данных к одинаковому типу данных, для проведения сравнения
            if($op && gettype($temp[0]) != gettype($temp[1])){
                if(is_string($temp[1])){
                    if(!is_null($temp[0]) || in_array($op, ['=','~','~~']))
                        $temp[0] = ''.$temp[0];
                } elseif($temp[1] === false || $temp[1] === true){
                    if(!is_null($temp[0])){
                        if($temp[0] == 'false') $temp[0] = false;
                        elseif($temp[0] == 'true') $temp[0]  = true;
                    }
                } elseif(!is_null($temp[0]) && !is_object($temp[0]))
                    $temp[0] = (float)$temp[0];
            }

            # сравнение данных, для фильтрации
            switch($op){
                case '=':  $result = $temp[0] == $temp[1]; break;
                case '~':  $result = strpos($temp[0], $temp[1]) !== false; break; //preg_match('/'.$temp[1].'/', $temp[0]); break;
                case '~~': $result = strpos(Str::lower($temp[0]), Str::lower($temp[1])) !== false; break; //preg_match('/'.$temp[1].'/i', $temp[0]); break;
                case '!':  $result = $temp[0] && $temp[0] != $temp[1]; break;
                case '>':  $result = $temp[0] && $temp[0] > $temp[1]; break;
                case '<':  $result = $temp[0] && $temp[0] < $temp[1]; break;
                case '>=': $result = $temp[0] && $temp[0] >= $temp[1]; break;
                case '<=': $result = $temp[0] && $temp[0] <= $temp[1]; break;
                default:
                    //dd($temp, $elem1);
                    $temp = $temp[0];
                    if(is_null($temp))
                        $result = false;
                    elseif($temp instanceof XmlNode)
                        $result = true;
                    elseif($elem1[0] == '@')
                        $result = $temp;
                    else
                        $result = $node->get_nodes_is_value($temp, null, false);
            }
            if($result)
                $resultElement[] = $node;
        }
        return $resultElement;
    }
}
