
Module.modify :UniversalXML do
	methods :class_BaseNode => %q{ #{}
		#
		# *.root
		#			:> User::UniversalXML::Node or nil
		# *.to_s
		#			:> String
		# *.last
		#			:> User::UniversalXML::Node
		# *.first
		#			:> User::UniversalXML::Node
		#			:> Array of User::UniversalXML::Node
		# *.odd nodes = nil
		#			<: nodes ~ nil or Array of User::UniversalXML::Node
		#			:> Array of User::UniversalXML::Node
		# *.even nodes = nil
		#			<: nodes ~ nil or Array of User::UniversalXML::Node
		#			:> Array of User::UniversalXML::Node
		# *.node? tagname
		#			<: tagname ~ Symbol or String
		#			:> Bool
		#
		# *.get_tags_name
		#			:> Array of String
		# *.get_nodes_by_tag _tag, children = true
		#			<: _tag ~ String or Symbol
		#			<: children ~ Bool
		#			:> Array of User::UniversalXML::Node
		# *.get_nodes_by_id _id, children = true
		#			<: _id ~ String
		#			<: children ~ Bool
		#			:> Array of User::UniversalXML::Node
		# *.get_nodes_is_attr param, children = true
		#			<: param ~ Any
		#			<: children ~ Bool
		#			:> Array of User::UniversalXML::Node
		# *.get_nodes_is_value value, param = nil, children = true
		#			<: value ~ Any
		#			<: param ~ nil or Any
		#			<: children ~ Bool
		#			:> Array of User::UniversalXML::Node
		#
		# *.xmlPath path, oldresult = nil, &block
		#			<: path ~ StringPath
		#			<: oldresult ~ nil or Array of User::UniversalXML::Node
		# *.match_find nodes, elem1, op = nil, elem2 = nil
		#			<: nodes ~ nil or Array of User::UniversalXML::Node
		#			<: elem1 ~ String
		#			<: op ~ nil or String of '~~', '~', '!', '<=', '>=', '>', '<', '='
		#			<: elem2 ~ nil or String
		#			:> Bool
		#
		# ================================
		# =          StringPath          =
		# ================================
		#
		# /       - разделитель уровней
		# &&      - разделитель поиска на одном уровне AND
		# ||      - разделитель поиска на одном уровне OR
		# .       - самый верхний узел
		# ..      - верхний узел
		# *       - обход всех подузлов или все узлы, если последний элемент
		# **      - рекурсивный обход всех подэлементов
		# *.      - все конечные элементы
		# **.     - все конечные элементы, рекурсивный обход
		# name    - узлы по тегу
		# #       - отсутствие id и name
		# #name   - узлы по id или name
		# #:name  - регистронезависимые узлы по id или name
		# @       - отсутствие атрибутов
		# @name   - аттрибут тега
		# @:name  - регистронезависимый аттрибут тега
		# :last   - последний элемент в результате
		# :first  - первый элемент в результате
		# :number - элемент результата
		# :odd    - нечетные элементы результата
		# :even   - четные элементы результата
		# :text   - текст хранящийся в теге
		# 'text'  - строка
		# :'text' - строка
		# 123     - число
		# true
		# false
		#
		# вычисления (где $ означает либо аттрибут либо спецсимвол/значение):
		# 
		# $a = $b   - если $a равен $b
		# $a ~ $b   - если в $a присутствует $b
		# $a ~~ $b  - если в $a присутствует $b (с игнорированием регистров)
		# $a ! $b   - если $a не равен $b
		# $a < $b   - если $a меньше $b
		# $a > $b   - если $a больше $b
		# $a <= $b  - если $a меньше или равен $b
		# $a >= $b  - если $a больше или равен $b
	
		class BaseNode
			# is_nodes?
			# parent
			# nodes
			# is_text?
			# is_attrs?
			# is_attr?
			# id
			# tag
			# full_tag
			# attrs
			# length
			# each
			# text

			def initialize
				@parse_value = false
			end

			def to_s
				"Node:<#{full_tag}[#{attrs(false).inject([]){|r,e| r << (e[0]+'="'+e[1]+'"')}.join(',')}] > ...#{length} elements"
			end

			def root
				# корневой узел (самый верхний узел XML-документа)
				r = parent || self
				r = r.parent until r.parent.nil?
				r
			end

			def odd _nodes = nil
				# возвращает все вложенные узлы, лежащие по нечетным индексам (1,3,5,7,...)
				(_nodes ? _nodes : nodes).select.with_index{|node, index| index.odd?}
			end
			
			def even _nodes = nil
				# возвращает все вложенные узлы, лежащие по четным индексам (0,2,4,6,...)
				(_nodes ? _nodes : nodes).select.with_index{|node, index| index.even?}
			end

			def last
				# возвращает последний вложенный узел
				nodes[-1]
			end
			
			def first
				# возвращает первый вложенный узел
				nodes[0]
			end
			
			def node? tagname
				# возвращает true, если вложенный узел с указанный именем тега существует, и false в противном случае
				tagname = tagname.to_s
				nodes.each{|node| return true if node.full_tag == tagname || node.tag == tagname}
				false
			end
			
			def tags_to_h
				# медот пробегает по всем вложенным узлам, и возвращает все конечные теги в виде хеша их имен и значений

				return {} unless @node.hasChildNodes()
				res = {}
				list = @node.getChildNodes()
				for num in 0..list.length-1
					node = list.item(num)
					if node.respond_to?(:getTagName) && !node.hasChildNodes()
						nm = node.getTagName.split(':')[-1].to_sym
						if node.respond_to? :getText
							txt = node.getText()
						else
							txt = node.getNodeValue()
							txt = node.getTextContent() if txt.nil?
						end
						res[nm] = txt
					end
				end
				res
			end

			def get_tags_name
				# возвращает все имена вложенных узлов/тегов
				nodes.inject([]){|result,node| if result.include?(node.tag) then result else result << node.tag end }
			end
			
			def get_nodes_by_tag _tag, children = true
				# возвращает все указанныые вложенные узлы, где
				#   _tag     - наименование тега, которые необходимо вернуть
				#   children - если true, то будет запущен рекурсивный поиск по всем вложенным узлам
				_tag = _tag.to_s
				result = nodes.select{|node| node.tag == _tag}
				each{|node| result += node.get_nodes_by_tag _tag} if children
				result
			end
			
			def get_nodes_by_id _id, children = true
				# возвращает все указанныые вложенные узлы, где
				#   _id      - чему должен быть равен атрибут "id" или "name" у тега, которые необходимо вернуть
				#   children - если true, то будет запущен рекурсивный поиск по всем вложенным узлам
				result = nodes.select{|node| node[:id] == _id}
				each{|node| result += node.get_nodes_by_id _id} if children
				result
			end
			
			def get_nodes_is_attr param, children = true
				# возвращает все указанныые вложенные узлы, где
				#   param    - наименование атрибута, который должен существовать у тега, которые необходимо вернуть
				#   children - если true, то будет запущен рекурсивный поиск по всем вложенным узлам
				result = nodes.select{|node| node.is_attr? param}
				each{|node| result += node.get_nodes_is_attr param} if children
				result
			end
			
			def get_nodes_is_value value, param = nil, children = true
				# возвращает все указанныые вложенные узлы, где
				#   value    - значение которому должен быть равен атрибут тега, которые необходимо вернуть
				#   param    - наименование атрибута, который нужно проверить у тега, которые необходимо вернуть,
				#              если указать как nil, то будет проверяться все атрибуты тега, на соответствие значению
				#   children - если true, то будет запущен рекурсивный поиск по всем вложенным узлам

				if param
					result = nodes.select{|node| node.is_attr?(param) && (@parse_value ? node[param] == value : node[param] == value.to_s)}
				else
					result = nodes.select{|node| node.attrs(false).inject(false){|res, elem| res || (@parse_value ? elem[1] == value : elem[1].to_s == value.to_s)} }
				end
				each{|node| result += node.get_nodes_is_value value, param} if children
				result
			end

		
			def xmlPath path, oldresult = nil, &block
				# возвращает все указанныые вложенные узлы, где
				#   path      - строка, которая является путем и описанием тегов, которые необходимо вернуть, подробную работу с ней смотрите в справочнике по модулю
				#   oldresult - массив узлов, из которых необходимо произвести дальнейшее отсеивание по указанной строке поиска
				#   &block    - если указан, то в него будет переданы поочередно все найденные узлы результата

				return [] if !is_nodes? && path[0]!='.'
				path = path.split('/') if path.class == String
				
				start = true
				temp = path.shift.split('&&').inject(oldresult) do |resultAND, sAND|
					tmp = sAND.split('||').inject([]) do |resultOR, sOR|
						resultOR + case sOR
						when '.'
							[root]
						when '..'
							resultAND.nil? ? [parent] : resultAND.map(&:parent)
						when '*'
							resultAND.nil? ? nodes : resultAND.map(&:nodes).flatten
						when '**'
							if resultAND.nil? 
								if is_nodes? 
									nds = nodes
									nds | nds.map{|node| node.xmlPath ['**'] }.flatten
								else
									[]
								end
							else
								resultAND.map{|node| node.xmlPath ['**']}.flatten
							end
						when '*.'
							if resultAND.nil? 
								is_nodes? ? nodes.select{|node| !node.is_nodes? } : []
							else
								resultAND.map{|node| node.xmlPath ['*.']}.flatten
							end
						when '**.'
							if resultAND.nil? 
								if is_nodes? 
									nds = nodes
									nds.select{|node| !node.is_nodes? } | nds.map{|node| node.xmlPath ['**.'] }.flatten 
								else 
									[]
								end
							else
								resultAND.map{|node| node.xmlPath ['**.']}.flatten
							end
						when ':last'
							[*(start ? (resultAND.nil? ? last : resultAND.map(&:last)) : resultAND[-1])].compact
						when ':first'
							[*(start ? (resultAND.nil? ? first : resultAND.map(&:first)) : resultAND[0])].compact
						when ':odd'
							odd(resultAND).flatten.uniq
						when ':even'
							even(resultAND).flatten.uniq
						else 
							args = [sOR]
							['~~','~','!','<=','>=','>','<','='].each do |op|
								unless sOR.index(op).nil?
									args = sOR.split(op).insert(1,op)
									break
								end
							end
							if start && resultAND
								resultAND.map{|node| node.match_find(nil, *args)}.flatten
							else
								match_find(resultAND, *args).flatten
							end
						end
					end
					start = false
					tmp
				end

				return [] if temp.empty?
				
				result = path.empty? ? temp.uniq : xmlPath(path, temp).flatten.uniq
				result.each{|elem| yield elem} if block_given?
				result
			end
			
			def match_find(_nodes, elem1, op = nil, elem2 = nil)
				# вспомогательный метод, который работает внутри xmlPath, и занимается сравниванием значений, подробнее можно почитать в справочнике по модулю
				#   _nodes - массив узлов, которые следует отфильтровать, по указанным правилам
				#   elem1  - правое правило фильтрации
				#   op     - оператор сравнения
				#     
				#   elem2  - левое правило фильтрации

				_nodes = nodes if _nodes.nil?
				return [] if nodes.empty?
				_nodes.select.with_index do |node, index|
					# получение значений по указанным правилам
					temp = [elem1,elem2].compact.map do |elem|
						if elem == ':number'
							index
						elsif elem == ':text'
							op ? node.text : (node.is_text? ? true : nil)
						elsif elem == 'true'
							true
						elsif elem == 'false'
							false
						elsif elem == '@'
							!node.is_attrs?
						elsif elem[0..1] == '@:'
							key = elem[2..-1]
							op ? (node.is_attr?(key, true) ? node[key.to_s.to_java.to_lower_case] : nil) : node.is_attr?(key, true)							
						elsif elem[0] == '@'
							key = elem[1..-1]
							op ? (node.is_attr?(key) ? node[key] : nil) : node.is_attr?(key)
						elsif elem == '#'
							node unless node.id
						elsif elem[0..1] == '#:'
							temp = (elem[2] == elem[-1] && elem[-1] == "'") ? elem[3...-1] : elem[2..-1]
							node if node.id && node.id.to_java.to_lower_case == temp.to_java.to_lower_case
						elsif elem[0] == '#'
							temp = (elem[1] == elem[-1] && elem[-1] == "'") ? elem[2...-1] : elem[1..-1]
							node if node.id == elem[1..-1]
						elsif elem.match(/^\d+(\.\d+)?$/)
							elem.to_f
						elsif elem[0] == elem[-1] && elem[0] == "'"
							elem[1...-1]
						elsif elem[0] == ':' && elem[1] == elem[-1] && elem[1] == "'"
							elem[2...-1].to_java.to_lower_case
						elsif node.tag == elem || node.full_tag == elem
							node
						else
							nil
						end
					end
					
					# предварительное преобрасование данных к одинаковому типу данных, для проведения сравнения
					if op && temp[0].class != temp[1].class
						if temp[1].class == String
							temp[0] = temp[0].to_s if !temp[0].nil? || ['=','~','~~'].include?(op)
						elsif [false,true].include? temp[1]
							unless temp[0].nil?
								temp[0] = false if temp[0] == 'false'
								temp[0] = true if temp[0] == 'true'
							end
						elsif temp[0].class != Node
							temp[0] = temp[0].to_f unless temp[0].nil?
						end
					end
					
					# сравнение данных, для фильтрации
					case op
					when '='
						temp[0] == temp[1]
					when '~'
						temp[0] =~ Regexp.new(temp[1])
					when '~~'
						temp[0] =~ Regexp.new(temp[1],Regexp::IGNORECASE)
					when '!'
						temp[0] && temp[0] != temp[1] 
					when '>'
						temp[0] && temp[0] > temp[1]
					when '<'
						temp[0] && temp[0] < temp[1]
					when '>='
						temp[0] && temp[0] >= temp[1]
					when '<='
						temp[0] && temp[0] <= temp[1]
					else
						temp = temp[0]
						if temp.nil?
							false
						elsif temp.class == self.class
							true
						elsif elem1[0] == '@'
							temp
						else
							node.get_nodes_is_value temp, nil, false
						end
					end
				end
			end
		end
	},
		:class_NodeAdapter => %q{ #{}
		class NodeAdapter < BaseNode
			# адаптер, для java-класса XML-документа

			def self.isNodes? _node
				return false unless _node.hasChildNodes()
				list = _node.getChildNodes()
				for num in 0..list.length-1
					return true if list.item(num).respond_to?(:getTagName)
				end
				false
			end

			def initialize org_w3c_dom_Node
				if org_w3c_dom_Node.is_a? org.w3c.dom.Node
					super()
					@node = org_w3c_dom_Node 
				else
					raise "Parametrs is not '#{org.w3c.dom.Node}' type"
				end
			end
			
			def node? tagname
				tagname = tagname.to_s
				if @node.hasChildNodes()
					list = @node.getChildNodes()
					for num in 0..list.length-1
						node = list.item(num)
						return true if node.respond_to?(:getTagName) && node.getTagName == tagname
					end 
				end
				return false
			end

			def is_nodes?
				#@node.hasChildNodes()
				NodeAdapter.isNodes? @node
			end

			def parent
				NodeAdapter.new @node.getParentNode()
			end

			def full_tag
				@node.getTagName()
			end

			def tag
				pref,name = full_tag.split(':')
				name || pref
			end

			def prefix
				@node.getNodePrefix()
			end

			def is_attrs?
				@node.hasAttributes()
			end

			def is_attr? name
				@node.hasAttribute name
			end

			def length
				@node.getChildNodes().length
			end

			def find_id _id
				res = @node.getDocument.getElementById(_id.to_s)
				res = NodeAdapter.new res if res
				res
			end
			
			def tags_to_h
				return {} unless @node.hasChildNodes()
				res = {}
				list = @node.getChildNodes()
				for num in 0..list.length-1
					node = list.item(num)
					if node.respond_to?(:getTagName) && !NodeAdapter.isNodes?(node)
						nm = node.getTagName.split(':')[-1].to_sym
						if node.respond_to? :getText
							txt = node.getText()
						else
							txt = node.getNodeValue()
							txt = node.getTextContent() if txt.nil?
						end
						res[nm] = txt
					end
				end
				res
			end

			def nodes
				res = []
				list = @node.getChildNodes()
				for num in 0..list.length-1
					node = list.item(num)
					res << NodeAdapter.new(node) if node.respond_to?(:getTagName)
				end 
				res
			end

			def id
				if @node.hasAttribute :id
					@node.getAttribute :id
				elsif @node.hasAttribute :name
					@node.getAttribute :name
				else
					nil
				end
			end

			def each &block
				return self if !block_given?
				list = @node.getChildNodes()
				for num in 0..list.length-1
					node = list.item(num)
					yield NodeAdapter.new(node) if node.respond_to?(:getTagName)
				end
				self
			end

			def text
				if @node.respond_to? :getText
					txt = @node.getText()
				else
					txt = @node.getNodeValue()
					txt = @node.getTextContent() if txt.nil?
				end
				txt.to_s
			end

			def is_text?
				!is_nodes? && !text.strip.empty?
			end

			def attrs keys=true
				result = keys ? [] : {}
				return result unless is_attrs? 
				list = @node.getAttributes()
				for num in 0..list.length-1
					attr = list.item(num)
					if keys
						result << attr.getName()
					else
						result[attr.getName.to_sym] = attr.getValue()
					end
				end
				result
			end
			
			
			public
			def to_text at_this = false, opts = nil, lvl = ''
				#opts:  [:cdata, :nopad, :small_end]
				opts = [:cdata, :small_end] unless opts.is_a?(Array)
				res = []
				
				tempFullTag = full_tag
				if at_this
					res << "#{lvl}<#{tempFullTag}"
					if @node.hasAttributes()
						res << ' '
						res << attrs(false).inject([]){|res,attr| res << " #{attr[0]}=\"#{attr[1]}\"" }.join('')
					end
				end
				isNodesEmpty = !is_nodes?
				isTextEmpty = !is_text?
				if at_this && isNodesEmpty && isTextEmpty && opts.include?(:small_end)
					res << '/>'
				elsif isNodesEmpty && isTextEmpty
					if at_this
						res << "></#{tempFullTag}>"
					else
						return ''
					end
				elsif isNodesEmpty
					res << '>' if at_this
					tempText = text
					res <<  if tempText[/[<>"'&]/] #"
										if opts.include? :cdata
											"<![CDATA[#{tempText}]]>"
										else
											tempText.tr('<>"\'&','')
										end
									else
										tempText
									end
					res << "</#{tempFullTag}>" if at_this
				else
					lvlnext = !at_this && opts.include?(:nopad) ? '' : lvl + "\t"
					res << ">\n" if at_this
					res << nodes.map{ |node| node.to_text(true, opts, lvlnext) }.join("\n")
					res << "\n#{lvl}</#{tempFullTag}>" if at_this
				end
				res.join()
			end
			
			def method_missing method, *args
				method = method.to_s
				if args.empty?
					nds = nodes
					#поиск по id
					nds.each{|node| return node if node.id == method }
					
					#поиск по имени тега
					nds.each{|node| return node if node.tag == method }
					
					#поиск параметра по имени
					return @node.getAttribute(method) if @node.hasAttribute method
					
					#=== регистронезависимый поиск ===
					methodL = method.to_java.to_lower_case
					
					
					nds.each{|node| id = node.id; return node if id && id.to_java.to_lower_case == methodL }
					
					#поиск по имени тега
					nds.each{|node| return node if  node.tag.to_java.to_lower_case == methodL }
					
					raise "Method, param, id and TAG '#{method}' not found"
				elsif args[0].is_a?(Hash) && args.length == 1
					nodes.each do |node| 
						if node.tag == method
							result = true
							params = node.attrs false
							args[0].each do |key, val| 
								key = key.to_s.to_sym
								result &&= params.key?( key )
								if result
									if val.is_a? Regexp
										result &&= params[key][val]
									else 
										result &&= params[key].to_s == val.to_s
									end
								end
							end
							return node if result
						end
					end
					
					raise "Method and TAG '#{method}' is filter '#{args[0]}' not found"
				else
					raise "Method '#{method}(#{args.map(&:inspect).join(',')})' not found"
				end
			end
		end
	},
	:class_Node => %q{ #{}
		UNIVERSAL_XML_FILE = "universal_settings_xml"
		
		# =================================
		# =        Node Interface:        =
		# =================================
		#
		# ---------------------------------
		# -        Constructors           -
		# ---------------------------------
		#
		# ...= User::UniversalXML::Node.new _tag, up = nil, _id = nil
		#			<: _tag ~ String or Symbol
		#			<: up ~ nil or User::UniversalXML::Node
		#			<: _id ~ nil or String
		# ...= User::UniversalXML::Node.load file_name
		#			<: file_name ~ String
		#
		# ---------------------------------
		# -          Data R/W             -
		# ---------------------------------
		#
		# *.tag
		#			:> String
		# *.prefix new_prefix = false
		#			<: new_prefix ~ false or String or Symbol or nil
		#			:> String
		# *.full_tag
		#			:> String
		# *.id
		#			:> String
		# *.obj_id
		#			:> String
		# *.is_attr? name, islow = false
		#			<: name ~ Any
		#			<: islow ~ Bool
		#			:> Bool
		# *.attrs keys = true
		#			<: keys ~ Bool
		#			:> Hash or Array of keys
		# *.attr key, value = nil
		#			<: key ~ Any
		#			<: value ~ Any
		#			:> self
		# *.attrs_count
		#			:> Numeric
		# *.is_attrs?
		#			:> Bool
		# *.[] key
		#			<: key ~ Any
		#			:> String
		# *.[]= key, value
		#			<: key ~ Any
		#			<: value ~ Any
		# *.text value = nil
		#			<: value ~ String
		#			:> String
		# *.text= value
		#			<: value ~ String
		# *.is_text?
		#			:> Bool
		# *.tag= newTag
		#			<: newTag ~ String or Symbol
		# *.attrs_parse children = true
		#			<: children ~ Bool
		#			:> self
		# *.attrs_to_s children = true
		#			<: children ~ Bool
		#			:> self
		# *.is_attr_parse?
		#			:> Bool
		#
		# ---------------------------------
		# -          Object Node          -
		# ---------------------------------
		#
		# *.clone children = true
		#			<: children ~ Bool
		#			:> User::UniversalXML::Node
		# *.move_to node
		#			<: node ~ User::UniversalXML::Node
		#			:> self
		# *.move_me_down step = 1
		#			<: step ~ :last or Number
		#			:> self
		# *.move_me_up step = 1
		#			<: step ~ :first or Number
		#			:> self
		# *.move_me step
		#			<: step ~ :first or :up or :down or :last or Number
		#			:> self
		# *.delete_me
		#			:> self
		# *.edit &block
		#			:> self
		# *.save file_name
		#			<: file_name ~ String
		#			:> self
		# *.to_xml encoding='UTF-8', version='1.0'
		#			<: encoding ~ nil or String
		#			<: version ~ String
		#			:> String
		# *.to_h
		#			:> Hash
		# *.tags_to_h
		#			:> Hash
		# *.to_text at_this = false, opts = nil, lvl = ''
		#			<: at_this ~ Bool
		#			<: opts ~ nil or Array of [:cdata, :esc, :nopad, :small_end]  # default = [:cdata, :esc]
		#			<: lvl ~ String
		#			:> String
		#
		# ---------------------------------
		# -          Tree Node            -
		# ---------------------------------
		#
		# *.parent
		#			:> User::UniversalXML::Node or nil
		# *.nodes
		#			:> Array of User::UniversalXML::Node
		# *.is_nodes?
		#			:> Bool
		# *.length
		#			:> Numeric
		# *.clear
		#			:> self
		# *.delete node
		#			<: node ~ User::UniversalXML::Node
		#			:> self
		# *.move_down node, step = 1
		#			<: node ~ Number or User::UniversalXML::Node
		#			<: step ~ :last or Number
		#			:> self
		# *.move_up node, step = 1
		#			<: node ~ Number or User::UniversalXML::Node
		#			<: step ~ :first or Number
		#			:> self
		# *.move node, step
		#			<: node ~ Number or User::UniversalXML::Node
		#			<: step ~ :first or :up or :down or :last or Number
		#			:> self
		# *.create_node name, _id = nil, &block
		#			<: name ~ String or Symbol
		#			<: _id ~ nil or String
		#			:> User::UniversalXML::Node
		# *.each &block
		#			:> self
		# *.size
		#			:> Integer
		# *.last_node path
		#			<: path ~ String
		#			:> self or User::UniversalXML::Node
		#
		# ---------------------------------
		# -     Find or Filter Nodes      -
		# ---------------------------------
		#
		# *.filter path = nil, &block
		#			<: path ~ nil or StringPath
		#			:> User::UniversalXML::Node
		# *.find_id _id
		#			<: _id ~ String
		#			:> nil or self or User::UniversalXML::Node
		#
		# ---------------------------------
		# -            Other              -
		# ---------------------------------
		#
		# *.index node
		#			<: node ~ User::UniversalXML::Node
		#			:> Integer
		# *.method_missing method, *args
		#			:> User::UniversalXML::Node or String or self or nil
		#
		

		class Node < BaseNode
			def initialize _tag, up = nil, _id = nil, &block
				super()
				_tag = _tag.to_s
				if _tag[':']
					@prefix, @tagName = _tag.split(':').map(&:to_sym)
				else 
					@prefix = nil
					@tagName = _tag.to_sym
				end
				@params = _id ? {:id => _id} : {}
				@nodes = []
				@parent = up
				@text = ''
				instance_eval(&block) if block_given?
			end

			def clone children = true
				res = Node.new @tagName
				res.attrs_parse if @parse_value
				res.text = @text
				@params.each{|key, val| res[key] = val}
				@nodes.each{|node| node.clone.move_to res} if children
				
				res
			end

			def tag
				@tagName.to_s
			end
			alias_method :тег, :tag
			
			def prefix new_prefix=false
				@prefix = new_prefix if !(new_prefix === false)
				@prefix.to_s
			end

			def full_tag
				if @prefix 
					"#{@prefix}:#{@tagName}"
				else
					@tagName.to_s 
				end
			end
			
			def tag= newTag
				if newTag.is_a?(Symbol)
					@tagName = newTag
				elsif newTag && !newTag.empty?
					if _tag[':']
						@prefix, @tagName = _tag.split(':').map(&:to_sym)
					else 
						@prefix = nil
						@tagName = _tag.to_sym
					end
				end
			end
			
			def id
				if @params.key? :id
					@params[:id]
				elsif @params.key? :name
					@params[:name]
				else
					nil
				end
			end
			alias_method :получить_id, :id
			alias_method :name, :id
			
			def obj_id
				result = 
				if @parent 
					path = @parent.obj_id
					@parent.index(self).to_s 
				else 
					path = ''
					''
				end
				(path.empty? ? '' : path + '/') + tag + (result.empty? ? '' : '#' + result)
			end
			
			def is_attr? name, islow = false
				if islow
					@params.keys.map{|key| key.to_s.to_java.to_lower_case}.include? name.to_s.to_java.to_lower_case
				else
					@params.key? name.to_s.to_sym
				end
			end
			alias_method :имеется_параметр?, :is_attr?
			
			def attrs keys = true
				keys ? @params.keys : @params
			end
			alias_method :параметры, :attrs
			
			def attrs_count
				@params.size
			end
			
			def is_attrs?
				!@params.empty?
			end

			def is_nodes?
				!@nodes.empty?
			end

			def is_text?
				!@text.empty?
			end

			def is_attr_parse?
				@parse_value
			end
			
			def nodes
				@nodes
			end
			alias_method :узлы, :nodes
			
			def [] key
				@params[key.to_s.to_sym]
			end
			
			def []= key, value
				@params[key.to_s.to_sym] = @parse_value ? value : value.to_s
			end

			def attrs_parse children = true
				@parse_value = true
				@params.each{|key,val| @params[key] = User::Utils.parse_value val}
				@nodes.each{|node| node.attrs_parse true} if children
				self
			end

			def attrs_to_s children = true
				@parse_value = false
				@params.each{|key,val| @params[key] = User::Utils.value_to_s val}
				@nodes.each{|node| node.attrs_to_s true} if children
				self
			end

			def parent
				@parent
			end
			
			def move node, step
				if node.is_a? Integer
					indx = node
					node = @nodes[node]
				else
					indx = @nodes.index node
				end
				@nodes.delete_at indx
				indx = case step
					when :first
						0
					when :last
						-1
					when :up
						indx > 0 ? indx - 1 : 0
					when :down
						indx < @nodes.size ? indx + 1 : -1
					else
						if step < 0
							step <= indx ? indx + step : 0
						elsif step > 0
							indx + step <= @nodes.size ? indx + step : -1
						else 
							indx
						end
				end
				@nodes.insert indx, node
				self
			end
			
			def move_up node, step = 1
				move node, step.is_a?(Integer) ? -step : step
				self
			end
			
			def move_down node, step = 1
				move node, step
				self
			end
			
			def move_me_up step = 1
				@parent.move_up self, step if @parent
				self
			end
			
			def move_me_down step = 1
				@parent.move_down self, step if @parent
				self
			end
			
			def move_me step
				@parent.move self, step
				self
			end
			
			def move_to node
				@parent.delete self if @parent
				@parent = node
				node.nodes << self
				self
			end
			alias_method :переместиться_в, :move_to
			
			def clear
				@nodes = []
				self
			end
			
			def delete node
				if node.is_a? Integer
					@nodes.delete_at node
				else
					@nodes.delete node
				end
				self
			end
			alias_method :удалить_узел, :delete
			
			def delete_me
				@parent.delete self if @parent
				self
			end
			alias_method :удалиться, :delete_me
			
			def attr key, value = nil
				@params[key.to_s.to_sym] = @parse_value ? value : value.to_s
				self
			end
			
			def text value = nil
				unless value.nil?
					@text = value.to_s.strip
				else
					if @nodes.empty?
						@text
					else
						to_text
					end
				end
			end
			
			def text= value
				@text = value.to_s.strip
			end
			
			def each &block
				@nodes.each(&block)
				self
			end
			
			def create_node name, _id = nil, &block
				node = Node.new(name, self, _id)
				@nodes << node
				node.instance_eval(&block) if block_given?
				node
			end
			alias_method :создать_узел, :create_node
			
			def text_node name, _text
				node = Node.new(name, self)
				node.text = _text
				@nodes << node
				node
			end
			
			def edit &block
				instance_eval(&block) if block_given?
				self
			end
			alias_method :редактировать, :edit
			
			# create_node :SOUNDS do
			#	create_node :STATION, "ЭМС.СЛ.Румянцево" do
			#		create_node :PRV, "ЭМС.СЛ.Румянцево.ВШ_613.Неисправность" do
			#			attr :sound, "Alarm"
			#		end
			#		create_node :PRV, "ЭМС.СЛ.Румянцево.ВШ_613.Печаль" do
			#			attr :sound, "Alarm"
			#		end
			#	end
			# end


			def filter path = nil, &block
				filterNodes = path.nil? ? @nodes : xmlPath(path)
				result = Node.new :FILTER
				filterNodes.each{|node| if !block_given? || yield(node) then node.clone.move_to result end}
				result
			end
						
			def length
				@nodes.length
			end
			alias_method :size, :length
			
			def last
				@nodes[-1]
			end
			
			def first
				@nodes[0]
			end
			
			def index node
				@nodes.index node
			end
			alias_method :поиск_узла, :index
			
			def find_id _id
				return self if @params.key?(:id) && @params[:id] == _id
				@nodes.each{ |node| return node if node[:id] == _id }
				nil
			end
			alias_method :поиск_по_id, :find_id
			
			def last_node path
				path = path.split('/') if path.class == String
				name = path.shift
				
				if name == '@'
					name = path.shift 
					_prefix = ''
					_prefix, name = name.split(':') if name[':']
					name = path.shift if name == @tagName.to_s && _prefix == @prefix.to_s
				end
				_prefix = ''
				_prefix, name = name.split(':') if name && name[':']
				return self unless name
				result = @nodes.select{|node|  node.тег == name && node.prefix == _prefix}[-1]
				result = result.last_node path if result && path.length > 0
				result || self
			end
			alias_method :последний_узел, :last_node
			
			alias_method :получить_теги_узлов, :get_tags_name
			alias_method :получить_узлы_по_тегу, :get_nodes_by_tag
			alias_method :получить_узлы_по_id, :get_nodes_by_id
			alias_method :получить_узлы_имеющие_параметр, :get_nodes_is_attr
			alias_method :получить_узлы_имеющие_значение, :get_nodes_is_value
			
			def to_xml encoding='UTF-8', version='1.0'
				this = self
				res = []
				res << "<?xml version=\"#{version}\" encoding=\"#{encoding}\"?>\n" unless encoding.nil?
				text = to_text true
				text = text.force_encoding(encoding) if text.encoding.to_s != encoding
				res << text
				res.join()
			end
			alias_method :получить_xml_текст, :to_xml
						
			def method_missing method, *args
				method = method.to_s
				if args.empty?
					#поиск по id
					@nodes.each{|node| return node if node.id == method }
					
					#поиск по имени тега
					@nodes.each{|node| return node if node.tag == method }
					
					#поиск параметра по имени
					return @params[method.to_sym] if @params.key? method.to_sym
					
					#=== регистронезависимый поиск ===
					methodL = method.to_java.to_lower_case
					
					
					@nodes.each{|node| id = node.id; return node if id && id.to_java.to_lower_case == methodL }
					
					#поиск по имени тега
					@nodes.each{|node| return node if  node.tag.to_java.to_lower_case == methodL }
					
					raise "Method, param, id and TAG '#{method}' not found"
				elsif args[0].is_a?(Hash) && args.length == 1
					@nodes.each do |node| 
						if node.tag == method
							result = true
							params = node.attrs false
							args[0].each do |key, val| 
								key = key.to_s.to_sym
								result &&= params.key?( key )
								if result
									if val.is_a? Regexp
										result &&= params[key][val]
									else 
										result &&= params[key].to_s == val.to_s
									end
								end
							end
							return node if result
						end
					end
					
					raise "Method and TAG '#{method}' is filter '#{args[0]}' not found"
				elsif args.length == 1
					if method['=']
						#поиск параметра по имени
						@params[method[0...-1].to_sym] = args[0]
					else
						@params[method.to_sym] = args[0]
						self
					end
				else
					raise "Method '#{method}(#{args.map(&:inspect).join(',')})' not found"
				end
			end

			def save file_name
				File.open(file_name, 'w+:utf-8') { |file| file.write to_xml }
				self
			end

			def to_h
				res = @params.clone
				res[:NODE_TEXT] = @text
				res
			end

			def tags_to_h
				res = {}
				@nodes.each{|node| res[node.tag.to_sym] = node.text if !node.is_nodes?}
				res
			end

			private
			def esc text, full = true
				text = @parse_value ? User::Utils.value_to_s(text) : text
				res = text.gsub('&','&amp;').gsub('<','&lt;').gsub('>','&gt;')
				if full
					res.gsub('"','&quot;').gsub("'",'&apos;')
				else
					res
				end
			end


			public
			def to_text at_this = false, opts = nil, lvl = ''
				#opts:  [:cdata, :esc, :nopad, :small_end]
				opts = [:cdata, :esc] unless opts.is_a?(Array)
				res = []
				if at_this
					res << "#{lvl}<#{full_tag}"
					if !@params.empty?
						res << ' '
						res << @params.inject([]){|res,attr| res << " #{attr[0]}=\"#{opts.include?(:esc) ? esc(attr[1]) : attr[1]}\"" }.join('')
					end
				end
				if at_this && @nodes.empty? && @text.empty? && opts.include?(:small_end)
					res << '/>'
				elsif @nodes.empty? && @text.empty? 
					if at_this
						res << "></#{full_tag}>"
					else
						return ''
					end
				elsif @nodes.empty?
					res << '>' if at_this
					res <<  if @text[/[<>"'&]/] #"
										if opts.include? :cdata
											"<![CDATA[#{@text}]]>"
										else
											esc @text, false
										end
									else
										@text
									end
					res << "</#{full_tag}>" if at_this
				else
					lvlnext = !at_this && opts.include?(:nopad) ? '' : lvl + "\t"
					res << ">\n" if at_this
					res << @nodes.map{ |node| node.to_text(true, opts, lvlnext) }.join("\n")
					res << "\n#{lvl}</#{full_tag}>" if at_this
				end
				res.join()
			end

			def self.load file_name
				res = nil
				File.open(file_name, 'r+:utf-8') { |file| res = User::UniversalXML.parse_xml file.read }
				res
			end
		end
	},
		:self_load => %q{ #{}
		def self.load file_name, baseNameTag = :CONFIG, desc = 'XML файл настроек'
			# на всякий случай хранить ссылку на binary не будем, мало ли
			binary = nil

			User::Binary.each do |id,bin|
				if bin.name == file_name && bin.file_ext == '.xml'
					binary = bin
					break
				end
			end
			if binary.nil?
				result = Node.new(baseNameTag)
				self.save file_name, result, desc, true
				result
			else
				self.parse_xml binary.text
			end
		end		
	},
		:self_save => %q{ #{}
		def self.save file_name, node, desc = nil, bin_id = nil
			text = node.to_xml
			unless bin_id
				User::Binary.each do |id,bin|
					if bin.name == file_name && bin.file_ext == '.xml'
						desc = bin.description unless desc
						bin_id = id
						break
					end
				end
			end
			bin_id = nil if bin_id === true
			desc = 'XML файл настроек' if desc.nil? || desc.empty? 
			file_hash = Digest::MD5.hexdigest(text).to_s
			byte_array = text.to_java_bytes.to_a
			descriptor = [bin_id,file_name,desc,file_name,".xml",byte_array.size,Time.now,file_hash,byte_array,nil]

			if descriptor[0] == nil
				$space.binaryAccount._insert *descriptor
			else
				$space.binaryAccount._update *descriptor
			end
		end
		
	},
		:self_load_xml_text => %q{ #{}
	
		require 'xml/sax_processor.rb'
		def self.load_xml_text text
			xml = StringIO.new(text)
			xmlNode = nil
			last_path = ''
			begin
				::SaxProcessor.new(xml) do
					match '*' do |whot|
						node do |node|
							if xmlNode
								last_path = node.prev_node.to_s
								xmlNode.last_node(last_path).create_node node.qname do
									node.attrs.each{|a| attr a.name, a.value}
								end
								last_path += '/' + node.qname
							else 
								xmlNode = User::UniversalXML::Node.new(node.qname)
								node.attrs.each{|a| xmlNode.attr a.name, a.value}
							end
						end
						
						text do |text|
							#puts last_path,text.size,'_______________'
							last = xmlNode.last_node(last_path)
							last = last.parent while !last.text.empty? && last.parent
							last.text = text
						end
					end
					
				end
			rescue Exception => e
				LogUtil.logger.error "#{inspect}.#{__method__} - ошибка при парсинге xml - \n#{e}"
			end
			xmlNode
		end
	},
		:self_handleNotification => %q{ #{}
		def self.handleNotification  ntf
			if ntf.handler == :binaryAccount && ntf.file_ext == '.xml'
				if ntf.action == Notifier::DELETE
					User::M_Events_Driver.exec_event self, :XML, "delete: #{ntf.name}", ntf
					User::M_Events_Driver.exec_event self, :XML, "delete file", ntf
				elsif ntf.action == Notifier::INSERT || ntf.action == Notifier::UPDATE
					User::M_Events_Driver.exec_event self, :XML, "update: #{ntf.name}", ntf
					User::M_Events_Driver.exec_event self, :XML, "update file", ntf
				end
			end
		end
	},
		:self_subscribe => %q{ #{}
		def self.subscribe
			$space[:Main].accounts[:binaryAccount].subscribe self, nil  if User.constants.include? :M_Events_Driver
		end
	},
		:self_unsubscribe => %q{ #{}
		def self.unsubscribe
			$space[:Main].accounts[:binaryAccount].unsubscribe self, nil
		end
	},
		:self_initialize => %q{ #{}
		def self.initialize
			subscribe
		end
	},
		:self_parse_xml => %q{ #{}
		def self.parse_xml text
			text = text.force_encoding('UTF-8') if text.encoding.to_s != 'UTF-8'
			tags = text.split('<')
			x = nil
			begin
			while !tags.empty? do
				tag = tags.shift
				sym = tag[0]
				case sym
					#<?...
					when '?'

					#<!...
					#<![CDATA[...
					#<!--...
					when '!'
						tag.strip!
						if tag[0..7] == '![CDATA[' 
							indx = -1
							tags.each_with_index{|a,i| if a.rindex(']]>') then indx = i; break end}
							tag = '<' + tags[0..indx].join('<')
							tags = tags[indx+1..-1]
							tag = tag[0...-3] if tag[-3..-1] == ']]>'
							if x
								x.text = tag
							else
								return tag
							end
						elsif tag[0..2] == '!--'
							tag = ''
							tag = tags.shift while !tags.empty? && tag[-3..-1] == '-->'
						end

					#</...
					when '/'
						return x if x.parent.nil?
						tag.strip!
						tag = tag[1...-1]
						x = x.parent while x.parent && x.full_tag != tag
						return x if x.parent.nil?
						x = x.parent

					else if tag.strip.empty?
						x.text = tag unless x.nil?
					else
						close = tag.rindex('>')
						if close.nil?
							indx = -1
							tags.each_with_index{|a,i| if a.rindex('>') then indx = i;break end}
							tag = tag + '<' + tags[0..indx].join('<')
							tags = tags[indx+1..-1]
						end
						tag.strip!
						close = tag.rindex('>')
						
						if close
							text = tag[close+1..-1]
							if tag[close-1] == '/'
								close -= 1
								close_node = true
							else 
								close_node = false
							end
							tag = tag[0...close].strip
						else
							text = ''
						end
							
						if tag.empty?
							x.text = text unless x.nil?
						else
							params = tag.split(' ')
							tagname = params.shift
							if !tagname.empty?
								x = x.nil? ? User::UniversalXML::Node.new(tagname) : x.create_node(tagname)
								x.text = text

								while !params.empty? do
									param = params.shift
									e = param.index('=')
									if e
										key = param[0...e]
										value = param[e+1..-1]
										if ['"',"'",'`'].include?(value[0]) && value[0] != value[-1]
											sym = value[0]
											value += ' ' + params.shift until ['"',"'",'`'].include?(value[-1]) && value[-1] == sym
										end
										value = value[1...-1] if value[0] == value[-1] && ['"',"'",'`'].include?(value[0])
										x[key] = value.gsub('&quot;','"').gsub('&apos;',"'").gsub('&lt;','<').gsub('&gt;','>').gsub('&amp;','&')
									else
										x[param] = param
									end
								end
							end
							x = x.parent if close_node
						end
					end
				end
			end
			rescue Exception => e
				LogUtil.logger.error "#{inspect}.#{__method__} - ошибка при парсинге xml - \n#{e}\nNoParse:\n#{tags.join('<')}"
			end
			x
		end
	},
		:self_load_from_file => %q{ #{}
		def self.load_from_file file_name
			Node.load file_name
		end
	},
		:self_save_to_file => %q{ #{}
		def self.save_to_file node, file_name
			node.save file_name
		end
		
		class << self
			alias_method :получить_файл, :load
			alias_method :обновить_файл, :save
			alias_method :openfile, :load_from_file
		end
	}
end
